const { PrismaClient } = require("@prisma/client");

const prisma = new PrismaClient();

async function main() {
  console.log("🌱 Adding incomplete item data to recurring tasks...");

  // Get all recurring tasks
  const tasks = await prisma.task.findMany({
    where: {
      isRecurring: true,
      recurrencePattern: { in: ["daily", "weekly", "monthly"] },
    },
    include: {
      assignedTo: true,
      createdBy: true,
    },
  });

  if (tasks.length === 0) {
    console.log(
      "❌ No recurring tasks found. Please create some recurring tasks first."
    );
    return;
  }

  const today = new Date();
  today.setHours(0, 0, 0, 0);

  for (const task of tasks) {
    console.log(`\n📅 Processing: ${task.title}`);
    console.log(`   Pattern: ${task.recurrencePattern}`);

    // Get all required elements for this task
    let requiredElementIds = [];

    if (task.scopeType === "element" && task.elementId) {
      requiredElementIds = [task.elementId];
    } else if (task.scopeType === "location" && task.locationId) {
      const selectedElements = task.selectedElementsJson
        ? JSON.parse(task.selectedElementsJson)
        : null;

      if (selectedElements && selectedElements.length > 0) {
        requiredElementIds = selectedElements;
      } else {
        const elements = await prisma.element.findMany({
          where: {
            locationId: task.locationId,
            isVisible: true,
          },
          select: { id: true },
        });
        requiredElementIds = elements.map((e) => e.id);
      }
    } else if (task.scopeType === "department" && task.departmentId) {
      const selectedLocations = task.selectedLocationsJson
        ? JSON.parse(task.selectedLocationsJson)
        : null;
      const selectedElements = task.selectedElementsJson
        ? JSON.parse(task.selectedElementsJson)
        : null;

      let locationIds = [];
      if (selectedLocations && selectedLocations.length > 0) {
        locationIds = selectedLocations;
      } else {
        const locations = await prisma.location.findMany({
          where: {
            departmentId: task.departmentId,
            isVisible: true,
          },
          select: { id: true },
        });
        locationIds = locations.map((l) => l.id);
      }

      if (selectedElements && selectedElements.length > 0) {
        const elements = await prisma.element.findMany({
          where: {
            id: { in: selectedElements },
            locationId: { in: locationIds },
            isVisible: true,
          },
          select: { id: true },
        });
        requiredElementIds = elements.map((e) => e.id);
      } else {
        const elements = await prisma.element.findMany({
          where: {
            locationId: { in: locationIds },
            isVisible: true,
          },
          select: { id: true },
        });
        requiredElementIds = elements.map((e) => e.id);
      }
    }

    if (requiredElementIds.length === 0) {
      console.log(`   ⏭️  Skipping - no elements found for this task`);
      continue;
    }

    console.log(`   Found ${requiredElementIds.length} required elements`);

    const assignedById = task.assignedToId || task.createdById;
    let addedCount = 0;
    let incompleteCount = 0;

    // Get existing item completions to see which periods already have data
    const existingCompletions = await prisma.taskItemCompletion.findMany({
      where: {
        taskId: task.id,
        periodDate: { not: null },
      },
      select: {
        periodDate: true,
        elementId: true,
      },
    });

    // Group existing completions by period
    const existingPeriods = new Map();
    existingCompletions.forEach((c) => {
      if (c.periodDate) {
        const periodKey = new Date(c.periodDate).toISOString().split("T")[0];
        if (!existingPeriods.has(periodKey)) {
          existingPeriods.set(periodKey, new Set());
        }
        existingPeriods.get(periodKey).add(c.elementId);
      }
    });

    // Determine how many past periods to process
    let numPastPeriods = 0;
    if (task.recurrencePattern === "daily") {
      numPastPeriods = 10; // Last 10 days
    } else if (task.recurrencePattern === "weekly") {
      numPastPeriods = 4; // Last 4 weeks
    } else if (task.recurrencePattern === "monthly") {
      numPastPeriods = 3; // Last 3 months
    }

    // Process past periods
    for (let i = numPastPeriods; i >= 1; i--) {
      let periodDate = new Date(today);

      if (task.recurrencePattern === "daily") {
        periodDate.setDate(periodDate.getDate() - i);
      } else if (task.recurrencePattern === "weekly") {
        periodDate.setDate(periodDate.getDate() - i * 7);
        const day = periodDate.getDay();
        const diff = periodDate.getDate() - day + (day === 0 ? -6 : 1);
        periodDate = new Date(periodDate.setDate(diff));
      } else if (task.recurrencePattern === "monthly") {
        periodDate.setMonth(periodDate.getMonth() - i);
        periodDate.setDate(1);
      }
      periodDate.setHours(0, 0, 0, 0);

      // Ensure we don't create completions before the task's start date
      const startDate = task.startDate || task.createdAt;
      if (startDate && periodDate < startDate) {
        continue;
      }

      const periodKey = periodDate.toISOString().split("T")[0];
      const existingForPeriod = existingPeriods.get(periodKey) || new Set();

      // For some periods, create incomplete item completions
      // Leave about 40% of periods with incomplete items (more aggressive)
      const shouldHaveIncompleteItems = i % 2 === 0 || i % 3 === 0;

      if (
        shouldHaveIncompleteItems &&
        existingForPeriod.size === requiredElementIds.length
      ) {
        // If all items are already completed for this period, delete some to create incomplete data
        const completionRate = 0.6 + Math.random() * 0.2; // 60-80%
        const numToKeep = Math.floor(
          requiredElementIds.length * completionRate
        );
        const elementsToKeep = requiredElementIds
          .sort(() => Math.random() - 0.5)
          .slice(0, numToKeep);
        const elementsToDelete = requiredElementIds.filter(
          (id) => !elementsToKeep.includes(id)
        );

        // Delete completions for elements that should be incomplete
        for (const elementId of elementsToDelete) {
          try {
            await prisma.taskItemCompletion.deleteMany({
              where: {
                taskId: task.id,
                elementId: elementId,
                periodDate: periodDate,
              },
            });
          } catch (error) {
            // Ignore errors
          }
        }

        incompleteCount++;
        console.log(
          `   ✅ Period ${periodDate.toLocaleDateString()}: Completed ${
            elementsToKeep.length
          }/${requiredElementIds.length} items (incomplete - removed ${
            elementsToDelete.length
          } completions)`
        );
      } else if (shouldHaveIncompleteItems) {
        // Complete only 60-80% of items for this period
        const completionRate = 0.6 + Math.random() * 0.2; // 60-80%
        const numToComplete = Math.floor(
          requiredElementIds.length * completionRate
        );
        const elementsToComplete = requiredElementIds
          .sort(() => Math.random() - 0.5)
          .slice(0, numToComplete);

        // Create completions for selected elements (only if not already completed)
        let createdForPeriod = 0;
        for (const elementId of elementsToComplete) {
          if (!existingForPeriod.has(elementId)) {
            try {
              await prisma.taskItemCompletion.create({
                data: {
                  taskId: task.id,
                  elementId: elementId,
                  checkedById: assignedById,
                  status: "OK",
                  periodDate: periodDate,
                  checkedAt: new Date(
                    periodDate.getTime() + 10 * 60 * 60 * 1000
                  ), // 10 AM
                },
              });
              addedCount++;
              createdForPeriod++;
            } catch (error) {
              // Item completion might already exist, skip
            }
          }
        }

        if (
          createdForPeriod > 0 ||
          elementsToComplete.length < requiredElementIds.length
        ) {
          incompleteCount++;
          console.log(
            `   ✅ Period ${periodDate.toLocaleDateString()}: Completed ${
              elementsToComplete.length
            }/${requiredElementIds.length} items (incomplete)`
          );
        }
      } else {
        // Complete all items for this period (only if not already completed)
        let createdForPeriod = 0;
        for (const elementId of requiredElementIds) {
          if (!existingForPeriod.has(elementId)) {
            try {
              await prisma.taskItemCompletion.create({
                data: {
                  taskId: task.id,
                  elementId: elementId,
                  checkedById: assignedById,
                  status: "OK",
                  periodDate: periodDate,
                  checkedAt: new Date(
                    periodDate.getTime() + 10 * 60 * 60 * 1000
                  ), // 10 AM
                },
              });
              addedCount++;
              createdForPeriod++;
            } catch (error) {
              // Item completion might already exist, skip
            }
          }
        }
        if (createdForPeriod > 0) {
          console.log(
            `   ✅ Period ${periodDate.toLocaleDateString()}: Completed all ${
              requiredElementIds.length
            } items`
          );
        }
      }
    }

    console.log(`   ✨ Summary:`);
    console.log(`      - Added ${addedCount} item completions`);
    console.log(`      - ${incompleteCount} periods with incomplete items`);
  }

  console.log("\n✨ Done! Incomplete item data added to recurring tasks.");
  console.log(
    "💡 Open a recurring task and click 'History' to see incomplete items!"
  );
}

main()
  .catch((e) => {
    console.error("Error adding incomplete items:", e);
    process.exit(1);
  })
  .finally(async () => {
    await prisma.$disconnect();
  });
